rm# Filename: R_Script.R
#
# Project: AISM Paper "On the usage of randomized p-values in the Schweder-Spjotvoll estimator"
# Authors: A. Hoang, T. Dickhaus
#
# Date: 10.12.2020
#
# R Version: 3.6.1
#
# Input data:
# Output data: Figure_1.pdf, Figure_5.pdf, Figure_6.pdf



# Definition / inclusion of functions and packages:

install.packages("latex2exp")

library(latex2exp)
source("functions/functions.R")

############
# Figure 1 #
############

n <- 50 # sample size in each study / null hypothesis

theta0 <-
  -1 / sqrt(n) # null parameter for the left subplot of the figure
theta1 <-
  1 / sqrt(n) # non-null parameter for the right subplot of the figure

# the figure
x <- 0:1000 / 1001

pdf(file = "results/Figure_1.pdf",
    width = 12,
    height = 6)

par(mfrow = c(1, 2))

plot(
  FLFC0,
  col = "red",
  ylab = TeX('probability $\\leq $ t'),
  xlab = TeX('$t$'),
  main = TeX(
    'Cdf of $p_{j}^{rand}$ for different choices of $c$ under the null hypothesis'
  ),
  lty = 5,
  lwd = 2
)
lines(
  x,
  Frand0(x, 0.25),
  col = "blue",
  type = "l",
  lty = 2,
  lwd = 2
)
lines(
  x,
  Frand0(x, 0.5),
  col = "black",
  type = "l",
  lty = 3,
  lwd = 2
)
lines(
  x,
  Frand0(x, 0.75),
  col = "orange",
  type = "l",
  lty = 4,
  lwd = 2
)
abline(0, 1, col = "green", lwd = 2)
legend(
  0,
  1,
  legend = c(
    TeX('$U_{j}$'),
    "c=0.25",
    "c=0.5",
    "c=0.75",
    TeX('p_{j}^{LFC}(X)')
  ),
  col = c("green", "blue", "black", "orange", "red"),
  lty = c(1, 2, 3, 4, 5),
  lwd = 2
)


plot(
  FLFC1,
  col = "red",
  ylab = TeX('probability $\\leq $ t'),
  xlab = TeX('$t$'),
  main = TeX(
    'Cdf of $p_{j}^{rand}$ for different choices of $c$ under the alternative'
  ),
  lty = 5,
  lwd = 2
)
lines(
  x,
  Frand1(x, 0.25),
  col = "blue",
  type = "l",
  lty = 2,
  lwd = 2
)
lines(
  x,
  Frand1(x, 0.5),
  col = "black",
  type = "l",
  lty = 3,
  lwd = 2
)
lines(
  x,
  Frand1(x, 0.75),
  col = "orange",
  type = "l",
  lty = 4,
  lwd = 2
)
abline(0, 1, col = "green", lwd = 2)
legend(
  0.55,
  0.45,
  legend = c(
    TeX('$U_{j}$'),
    "c=0.25",
    "c=0.5",
    "c=0.75",
    TeX('p_{j}^{LFC}(X)')
  ),
  col = c("green", "blue", "black", "orange", "red"),
  lty = c(1, 2, 3, 4, 5),
  lwd = 2
)

dev.off()

############
# Figure 5 #
############

pi0 <- 0.7 # proportion of the true null hypotheses
false <- 1 - pi0


m <- 1000 # number of null hypotheses
n <- 50 # sample size in each study
m0 <- round(m * pi0)

mu0 <- -1 / sqrt(n) # null parameter
mu1 <- 2.5 / sqrt(n) # non-null parameter

lambda <- 1 / 2 # lambda in the Schweder-Spjotvoll estimator

mu <- c()

for (i in 1:m) {
  if (i <= m0) {
    mu[i] <- mu0
  } else {
    mu[i] <- mu1
  }
}

set.seed(1, kind = NULL, normal.kind = NULL)


# one realization of LFC p-values
pLFC <- 1 - pnorm(sqrt(n) * rnorm(m, mu, 1 / sqrt(n)))


# relevant points for g, cf. paper
auxpts <- c(pLFC, 1 / lambda * pLFC[pLFC / lambda <= 1])

# the figure
pdf(file = "results/Figure_5.pdf",
    width = 6,
    height = 5.5)

plot(auxpts,
     sapply(auxpts, g),
     ylab = TeX('$g_{x}(\\lambda,c)$'),
     xlab = "c")

c <- auxpts[order(sapply(auxpts, g))[length(auxpts)]]
abline(v = c)
axis(
  side = 1,
  at = c,
  labels = TeX('$c_{0}$'),
  tck = -0.0275
)

dev.off()

############
# Figure 6 #
############

pi0 <- 0.7 # proportion of the true null hypotheses
false <- 1 - pi0


m <- 1000 # number of null hypotheses
n <- 50 # sample size in each study
m0 <- round(m * pi0)

mu0 <- -1 / sqrt(n) # null parameter
mu1 <- 2.5 / sqrt(n) # non-null parameter

lambda = 1 / 2 # lambda in the Schweder-Spjotvoll estimator

mu <- c()

for (i in 1:m) {
  if (i <= m0) {
    mu[i] <- mu0
  } else {
    mu[i] <- mu1
  }
}

set.seed(1, kind = NULL, normal.kind = NULL)

# one realization of LFC p-values
pLFC <- 1 - pnorm(sqrt(n) * rnorm(m, mu, 1 / sqrt(n)))

# corresponding realization of rand. p-values
prand <- c()
u <- runif(m)
for (i in 1:m) {
  if (pLFC[i] > c) {
    prand[i] <- u[i]
  } else{
    prand[i] <- 1 / c * pLFC[i]
  }
}

# Schweder-Spjotvoll estimations
pi0LFC <- (1 - ecdf(pLFC)(lambda)) / (1 - lambda)
pi0rand <- (1 - ecdf(prand)(lambda)) / (1 - lambda)


# the figure
pdf(file = "results/Figure_6.pdf",
    width = 6,
    height = 5.5)

plot(
  ecdf(pLFC),
  col = "red",
  #main=TeX('Ecdfs of $(p_{j}^{LFC})_{j=1,...,m}$ and $(p_{j}^{rand})_{j=1,...,m}$ under Model $1$'),
  xlab = TeX('$t$'),
  ylab = TeX('proportion$\\leq t$'),
  las = 1,
  xlim = c(0, 1),
  ylim = c(0, 1),
  main = NULL,
  xaxs = "i",
  yaxs = "i"
)
plot(ecdf(prand), col = "blue", add = TRUE)
abline(false, 1 - false, lwd = 2)
text(0.6, ecdf(prand)(0.6) - 0.05, TeX('rand'))
text(0.8, ecdf(pLFC)(0.8) - 0.05, TeX('LFC'))
abline(v = lambda, lty = 2)
abline(v = c, lty = 3)
abline(v = lambda * c, lty = 3)
abline(1 - pi0rand, pi0rand)
abline(1 - pi0LFC, pi0LFC)
axis(
  side = 1,
  at = c(lambda, c, lambda * c),
  labels = c(TeX('$\\lambda$'), TeX('$c_{0}$'), TeX('$\\lambda c_{0}$')),
  tck = -0.0275
)
axis(
  side = 2,
  at = c(false, 1 - pi0rand, 1 - pi0LFC),
  labels = c(
    TeX('$1-\\pi_{0}$'),
    TeX('$1-\\hat{\\pi}_{0}^{rand}(\\lambda)$'),
    TeX('$1-\\hat{\\pi}_{0}^{LFC}(\\lambda)$')
  ),
  tck = -0.0275,
  las = 1
)

dev.off()