% Filename: R_Script.R
% 
% Project: AISM Paper "On the usage of randomized p-values in the 
% Schweder-Spjotvoll estimator"
% Authors: A. Hoang, T. Dickhaus
% 
% Date: 10.12.2020
% 
% MATLAB Version: R2019a
% 
% Input data:
% Output data: 
%
% Inclusion of functions:
addpath 'functions';


%%%%%%%%%%%%
% Figure 2 %
%%%%%%%%%%%%

n = 50; % sample size in each study / null hypothesis

lambda = 1/2; % lambda in the Schweder-Spjotvoll estimator

mu0 = -1/sqrt(n); % null parameter
mu1 = 2.5/sqrt(n); % non-null parameter

pi0 = 0.7; % the proportion of the true null hypotheses

figure(1)

subplot(1,2,1);

f = @(x) (1-(pi0*Q2(lambda,mu0,x,n)+ ...
    (1-pi0)*Q2(lambda,mu1,x,n)))/(1-lambda);
fplot(f, [0,1]);
x = fminbnd(f,0,1);
xline(x, '--');
yline(f(x),'--');
xline(1-pi0);
yline(pi0,':');
xlabel('c');
ylabel('$h_{\theta}(1/2)(c)$', 'Interpreter', 'LaTeX');

subplot(1,2,2);

f = @(x) (1-(pi0*Q2(lambda,0,x,n)+ ...
    (1-pi0)*Q2(lambda,mu1,x,n)))/(1-lambda);
fplot(f, [0,1]);
x = fminbnd(f,0,1);
xline(x, '--');
yline(f(x),'--');
xline(1-pi0);
yline(pi0,':');
xlabel('c');
ylabel('$h_{\theta}(1/2)(c)$', 'Interpreter', 'LaTeX');

saveas(gcf,'results\Figure_2.pdf')

%%%%%%%%%%%%
% Figure 3 %
%%%%%%%%%%%%

m = 100; % number of null hypotheses

n = 50; % sample size in each study / null hypothesis

mu0 = -0.5/sqrt(n); % null parameter
mu1 = 1.5/sqrt(n); % non-null parameter
pi0 = 0.8; % the proportion of the true null hypotheses
lambda = 0.5; % lambda in the Schweder-Spjotvoll estimator

figure(2)
% bias plot

subplot(1,3,1);

bias = @(x) (1-(pi0*Q2(lambda,mu0,x,n)+ ...
    (1-pi0)*Q2(lambda,mu1,x,n)))/(1-lambda)-pi0;
fplot(bias, [0,1]);
xlabel('c');
ylabel('bias(c)', 'Interpreter', 'LaTeX');


% variance plot

subplot(1,3,2);

variance = @(x)  var(x,mu0,mu1,lambda,m,pi0,n,0,sigma);
fplot(variance, [0,1]);
xlabel('c');
ylabel('var(c)', 'Interpreter', 'LaTeX');

% mse plot

subplot(1,3,3);

mse = @(x) bias(x)^2+variance(x);
fplot(mse, [0,1]);
xlabel('c');
ylabel('MSE(c)', 'Interpreter', 'LaTeX');

saveas(gcf,'results\Figure_3.pdf')

%%%%%%%%%%%%
% Figure 4 %
%%%%%%%%%%%%

lambda = 0.5; % lambda in the Schweder-Spjotvoll estimator

m = 100; % number of null hypothesis
n = 50; % sample size in each study / null hypothesis

pi0 = 0.8; % the proportion of the true null hypotheses

sigma = 1/sqrt(n); % the standard deviation of the test statistics in the 
% multiple Z-tests model; cf. Example 1 in Section 2 in the paper
cv = 0.5/n; % the pairwise covariance of the test statistics  

mu0 = -0.5/sqrt(n); % the null parameter
mu1 = 1.5/sqrt(n); % the non-null parameter

figure(3)

% bias plot

subplot(1,3,1);

bias = @(x) (1-(pi0*Q2(lambda,mu0,x,n)+ ...
    (1-pi0)*Q2(lambda,mu1,x,n)))/(1-lambda)-pi0;
fplot(bias, [0,1]);
xlabel('c');
ylabel('bias(c)', 'Interpreter', 'LaTeX');


% variance plot

subplot(1,3,2);

% var = @(x) (1/(1-lambda)^2)*(1/m*(pi0*(Q2(lambda,mu0,x,n)-Q2(lambda,mu0,x,n)^2)+(1-pi0)*(Q2(lambda,mu1,x,n)-Q2(lambda,mu1,x,n)^2)) ... 
%     +1/m^2*(2*nchoosek(m*pi0,2)*cov(x,mu0,mu0,lambda,n,cv,sigma) ... 
%     +2*nchoosek(m-m*pi0,2)*cov(x,mu1,mu1,lambda,n,cv,sigma) ... 
%     +2*m^2*pi0*(1-pi0)*cov(x,mu0,mu1,lambda,n,cv,sigma)));

variance = @(x) var(x,mu0,mu1,lambda,m,pi0,n,cv,sigma);

fplot(variance, [0,1]);
xlabel('c');
ylabel('var(c)', 'Interpreter', 'LaTeX');


% mse plot

subplot(1,3,3);

mse = @(x) bias(x)^2+variance(x);
fplot(mse, [0,1]);
xlabel('c');
ylabel('MSE(c)', 'Interpreter', 'LaTeX');

saveas(gcf,'results\Figure_4.pdf')



%%%%%%%%%%%%%%
% Figure 7-9 %
%%%%%%%%%%%%%%

m = 100; % number of null hypothesis
n = 1; % sample size in each study / null hypothesis

pi0 = 0.75; % proportion of the true null hypotheses

alpha = 0.05; % significance level (FDR)

B = 100000; % number of Monte-Carlo repetitions

rng('default')

U = rand(B,m);

thresholds = alpha/m.*repmat(m:-1:1,B,1);

FDR = zeros(4,3,4,6);
power = zeros(4,3,4,6);
pi0estimate = zeros(4,3,4,6);

kmax = 10;
x = 1:kmax;
figure(4)
figure(5)
figure(6)
figure(7)
for i = 1:2
    for j = 1:2
        cv = (i-1)*0.5/n+(j-1)*0.25/n;
        Sigma = cv*ones(m);
        Sigma(1:m+1:end) = ones(1,m)/n;
        
        pi0estimate1 = zeros(1,kmax);
        pi0estimate2 = zeros(1,kmax);
        pi0estimate3 = zeros(1,kmax);
        
        FDR1 = zeros(1,kmax);
        FDR2 = zeros(1,kmax);
        FDR3 = zeros(1,kmax);
        
        power1 = zeros(1,kmax);
        power2 = zeros(1,kmax);
        power3 = zeros(1,kmax);
        
        value1 = zeros(1,kmax);
        value2 = zeros(1,kmax);
        value3 = zeros(1,kmax);
        
        
        for k = 1:kmax
            mu0 = -k*0.2/sqrt(n);
            mu1 = (1+k*0.25)/sqrt(n);
            
            lambda = 1/2;
            
            bias = @(x) (1-(pi0*Q2(lambda,mu0,x,n)+(1-pi0)*Q2(lambda,mu1,x,n)))/(1-lambda)-pi0;        
            
            cstar = fminbnd(bias,0,1);
            cdefault = 0.5;
            
            mu = [repmat(mu0,1,round(m*pi0)) repmat(mu1,1,m-round(m*pi0))];
            X = mvnrnd(mu,Sigma,B);
            
            p = 1-normcdf(X.*sqrt(n));
        
            pranddefault = U.*(p > cdefault) + p./cdefault.*(p <= cdefault);
            prandstar = U.*(p > cstar) + p./cstar.*(p <= cstar);


            Gp = (1-lambda)./(mean(p > lambda,2)+1/m);
            Gpranddefault = (1-lambda)./(mean(pranddefault > lambda,2)+1/m);
            Gprandstar = (1-lambda)./(mean(prandstar > lambda,2)+1/m);
            
            
            pi0estimate1(k) = mean(1./Gp);
            pi0estimate2(k) = mean(1./Gpranddefault);
            pi0estimate3(k) = mean(1./Gprandstar);
            
                   
            thresholdsGp = Gp.*thresholds;
            thresholdsGpranddefault = Gpranddefault.*thresholds;
            thresholdsGprandstar = Gprandstar.*thresholds;
            thresholdsoracle = 1/pi0.*thresholds;


            comparisonp = sort(p,2,'descend') < thresholdsGp;
            comparisonpranddefault = sort(p,2,'descend') < thresholdsGpranddefault;
            comparisonprandstar = sort(p,2,'descend') < thresholdsGprandstar;
            comparisonporacle = sort(p,2,'descend') < thresholdsoracle;


            [max1,index1] = max(comparisonp,[],2);
            [max2,index2] = max(comparisonpranddefault,[],2);
            [max3,index3] = max(comparisonprandstar,[],2);
            [max4,index4] = max(comparisonporacle,[],2);


            rip = max1.*index1;
            ripranddefault = max2.*index2;
            riprandstar = max3.*index3;
            riporacle = max4.*index4;


            [~,indices] = sort(p,2,'descend');
            [~,ranks] = sort(indices,2);


            rejectionp = ranks >= rip;
            rejectionpranddefault = ranks >= ripranddefault;
            rejectionprandstar = ranks >= riprandstar;
            rejectionporacle = ranks >= riporacle;


            FDRp = sum(rejectionp(:,1:round(m*pi0)),2)./(m-rip+1).*max1;
            FDRpranddefault = sum(rejectionpranddefault(:,1:round(m*pi0)),2)./(m-ripranddefault+1).*max2;
            FDRprandstar = sum(rejectionprandstar(:,1:round(m*pi0)),2)./(m-riprandstar+1).*max3;
            
            
            FDR1(k) = mean(FDRp);
            FDR2(k) = mean(FDRpranddefault);
            FDR3(k) = mean(FDRprandstar);
            

            powerp = sum(rejectionp(:,(round(pi0*m)+1):m),2)./(m-round(pi0*m)).*max1;
            powerpranddefault = sum(rejectionpranddefault(:,(round(pi0*m)+1):m),2)./(m-round(pi0*m)).*max2;
            powerprandstar = sum(rejectionprandstar(:,(round(pi0*m)+1):m),2)./(m-round(pi0*m)).*max3;
            powerporacle = sum(rejectionporacle(:,(round(pi0*m)+1):m),2)./(m-round(pi0*m)).*max4;
            
            
            power1(k) = mean(powerp)/mean(powerporacle);
            power2(k) = mean(powerpranddefault)/mean(powerporacle);
            power3(k) = mean(powerprandstar)/mean(powerporacle);
            
            
            value1(k) = mean(powerp)-mean(FDRp)/alpha;
            value2(k) = mean(powerpranddefault)-mean(FDRpranddefault)/alpha;
            value3(k) = mean(powerprandstar)-mean(FDRprandstar)/alpha;
            
            if (k == kmax)
                figure(5)
                subplot(2,2,(i-1)*2+j);
                scatter(x,FDR1,'k')
                hold on
                scatter(x,FDR3,'x','k')
                yline(alpha,'--');
                xlim([0.75 kmax+0.25]);
                if ((i-1)*2+j == 2)
                    legend('LFC','$c^{\star}$','Interpreter','latex');
                end
                title(['$\rho=$' num2str(cv)],'Interpreter','latex');
                sgtitle('FDR')
                xticks(1:kmax)
                xlabel('r')
                hold off
            end
            
            saveas(gcf,'results\Figure_8.pdf')

            
            if (k == kmax)
                figure(6)
                subplot(2,2,(i-1)*2+j);
                scatter(x,power1,'k')
                hold on
                scatter(x,power3,'x','k')
                yline(1,'--');
                xlim([0.75 kmax+0.25]);
                if ((i-1)*2+j == 2)
                    legend('LFC','$c^{\star}$','Interpreter','latex');
                end
                title(['$\rho=$' num2str(cv)],'Interpreter','latex');
                sgtitle('Relative powers')
                xticks(1:kmax)
                xlabel('r')
                hold off
            end
            
            if (k == kmax)
                figure(7)
                subplot(2,2,(i-1)*2+j);
                scatter(x,value1,'k')
                hold on
                scatter(x,value3,'x','k')
                xlim([0.75 kmax+0.25]);
                if ((i-1)*2+j == 2)
                    legend('LFC','randstar','Interpreter','latex');
                end
                title(['$\rho=$' num2str(cv)],'Interpreter','latex');
                sgtitle('Performances')
                xticks(1:kmax)
                xlabel('r')
                hold off
            end
            
            saveas(gcf,'results\Figure_9.pdf')
            
        end
    end
end



figure(4)
scatter(x,pi0estimate1,'k')
hold on
scatter(x,pi0estimate3,'x','k')
                
yline(pi0,'--');
xlim([0.75 kmax+0.25]);
legend('LFC','$c^{\star}$','Interpreter','latex');

sgtitle('$\pi_{0}$ estimations','Interpreter','latex')
xticks(1:kmax)                
xlabel('r')
hold off

saveas(gcf,'results\Figure_7.pdf')

